/*
 *  Copyright (c) 1992 John E. Davis  (davis@amy.tch.harvard.edu)
 *  All Rights Reserved.
 */
#include <stdio.h>
#include <string.h>

#include "slang.h"
#include "buffer.h"
#include "screen.h"
#include "window.h"
#include "ins.h"
#include "ledit.h"
#include "cmds.h"
#include "line.h"
#include "paste.h"
#include "display.h"
#include "vterm.h"
#include "sysdep.h"
#include "text.h"
#include "file.h"
#include "misc.h"
#include "search.h"

int Last_Command_Char;
int Blink_Flag = 1;

static char *Top_Of_Buffer_Error = "Top Of Buffer.";
static char *End_Of_Buffer_Error = "End Of Buffer.";

/* return 1 if line extends beyond the screen */
int long_line(void)
{
    int p, col;

    col = Screen_Col;
    p = Point;
    Point = CLine->len - 1;
    if (*(CLine->data + Point) != '\n') Point++;
    if (calculate_column() >= Window->width - 1)
      {
	  Point = p;
	  Screen_Col = col;
	  return(1);
      }
    Screen_Col = col;
    Point = p;
    return(0);
}

#define FAST_NOT_OK ((Repeat_Factor != NULL) || Window->trashed\
        || (ch < ' ') || (ch > 126)\
        || (Screen_Col >= Window->width - 2) || (Window->column > 1)\
	|| input_pending(2, 0) || long_line() || tabs_present())

/* This routine moves CLine and point to the matching '{' or whatever.  It
   returns 1 if it lies outside the screen, 0 if inside and -1 if not found
   and -2 if it looked back for more than 100 lines and not found.  It
   can get messed up if there are delimeters in comments unless they are
   properly enclosed like the above */

/* count is the number of lines to go back and shift is distance back to
   start searching from. (usually 0, 1, or 2) */
int find_matching(char ch, int count, int shift)
{
   int n = 1, slash, ret = 0, lang = ! (CBuf->modes == WRAP_MODE);
   unsigned char *p, *prev, *p1;
   char ch1, in, save;
   char quote = '\\';
   char cb1 = '/', cb2 = '*', ce1 = '*', ce2  = '/';
   int inc = 0;
   

    ch1 = '(';
    if (ch == '}') ch1 = '{'; else if (ch == ']') ch1 = '[';

    p = CLine->data + Point - shift;
    if (p < CLine->data)
      {
          if (CLine->prev == NULL) return(-1);
          if (CLine == Window->beg.line) ret = 1;
          CLine = CLine->prev;
	  p = CLine->data + (CLine->len - 1);
       }
     in = 0;
     if (Point > 1) prev = p - 1; else prev = p;

    while(count)
      {
	 p1  = p - 1;
	 if (lang && !inc && ((*p == '\'') || (*p == '"')))
	   {
	      slash = 0;
	      save = *p;
	      while((p > CLine->data) && (*(p - 1) == quote))
		{
		   p--;
		   slash = !slash;
		}

	      if (!slash)
		{
		   if (in == 0) in = save; else if (in == save) in = 0;
		}
	   }
	 
	      
          if (!in)
            {
	       /* This is specific to C.  I need to generalize to other languages. */
	       if (lang && (ce2 == *p) && (p > CLine->data)
		   && (*p1 == ce1))
		 {
		    p = p1;
		    inc = 1;
		 }
	       else if (lang && (cb2 == *p) && (p > CLine->data) 
		   && (*p1 == cb1))
		 {
		    if (inc == 0) return (-2);
		    p = p1;
		    inc = 0;
		 }
	       
	       if (!inc && ((*p == '[') || (*p == '{') || (*p == '('))) n--;
	       else if (!inc && ((*p == '}') || (*p == ']') || (*p == ')'))) n++;
            }

          if (!n) break;
          if ((p == CLine->data) && (CLine == Window->beg.line)) ret = 1;
          if (p == CLine->data)
            {
                if (CLine->prev == NULL) break;
                CLine = CLine->prev;
                count--;
                p = CLine->data + (CLine->len - 1);
            }
          else p--;
      }
    Point = (int) (p - CLine->data);
    if ((n == 0) && (*p == ch1)) return(ret);
    if (lang) if ((*prev == '\'') || (*prev == '"')) return(-2);
    return(-1);
}

/* blink the matching fence.  This assumes that the window is ok */
void blink_match(char ch)
{
   Line *save;
   int pnt, code, matchp;
   char buf[600];

   if (!Blink_Flag || (Repeat_Factor != NULL) || Window->trashed) return;
   pnt = Point;
   save = CLine;
   code = find_matching(ch, 100, 2);
   if (code == -1)
     {
	if ((! (CBuf->modes == WRAP_MODE)) && (!IS_MINIBUFFER)) message("Mismatch??");
     }
   else if (code == 0)
     {
	point_cursor();
	fflush(stdout);
	input_pending(2,1);   /* sleep((unsigned) 1); */
	Point = pnt;
	CLine = save;
	point_cursor();
	return;
     }
   else if (code == 1)
     {
	matchp = Point;
	Point = 0;
	strcpy(buf, "Matches ");
	skip_whitespace();
	if ((matchp == Point) && prevline(&Number_One))
	  {
	     Point = 0;
	     strcat(buf, buffer_substring());
	     nextline(&Number_One);
	     Point = 0;
	  }
	strcat(buf, buffer_substring());
	message(buf);
     }
   Point = pnt;
   CLine = save;
}

int goto_match()
{
    char *p;
    Line *save;
    int pnt, code;

    p = (char *) (CLine->data + Point);
    if ((*p != ')') && (*p != '}') && (*p != ']')) return(0);
    save = CLine;
    pnt = Point;
    code = find_matching(*p, -1, 1);  /* -1 since we want to shif back 1 */
    if (code == -1)
      {
	 if (!IS_MINIBUFFER) msg_error("Mismatch!!");
	 CLine = save;
	 Point = pnt;
	 return(0);
      }
    return(1);
}

int newline()
{
    int push = 0;

    if (bolp() && (CLine->prev != NULL))
      {
          push = 1;
          push_spot();
          CLine = CLine->prev;
	  Point = 0;
          eol();
      }

    split_line();
    ins('\n');
    if (push)
      {
          pop_spot();
          return(1);
      }

    Point--;
    forwchars(&Number_One);
    return(1);
}

int previous_char_cmd()
{
   int pnt;
   Cursor_Motion = 1;
   if (bobp())
     {
	msg_error(Top_Of_Buffer_Error);
	return(0);
     }

   pnt = Point - 1;
   backwchars(&Number_One);
   Goal_Column = calculate_column();
   return((pnt == -1) || Window->trashed);
}

/* Slang calls by reference so make this a pointer ! */
void insert_whitespace(int *n)
{
   int tab = User_Vars.tab;
   int c1, c2, i, k, nspace;

   if ((nspace = *n) <= 0) return;
   c1 = calculate_column() - 1;
   c2 = c1 + nspace;

   if (tab)
     {
	i = c1 / tab;
	k = c2 / tab - i;
	if (k) nspace = c2 - (i + k) * tab;
	ins_char_n_times('\t', k);
     }
   ins_char_n_times(' ', nspace);
}

/* check from point to end of line looking for tabs */
int tabs_present(void)
{
    unsigned char *p, *pmax;

    if (!User_Vars.tab) return(0);
    pmax = CLine->data + CLine->len;
    p = CLine->data + Point;

    while (p < pmax) if (*p++ == '\t') return(1);
    return(0);
}

int delete_char_cmd()
{
    int upd;
    char ch;

    if (eobp())
      {
          msg_error(End_Of_Buffer_Error);
          return(0);
      }

    ch = *(CLine->data + Point);

    if (FAST_NOT_OK)
      {
          del();
          upd = 1;
      }
    else
      {
          fast_del();
          tt_delete_char();
          upd = 0;
      }

    return(upd);
}

int backward_delete_char_cmd()
{
    char ch;

    if (bobp())
      {
          msg_error(Top_Of_Buffer_Error);
          return(0);
      }

    if (bolp())
      {
          backwchars(&Number_One);
          del();
          return(1);
      }

    ch = *(CLine->data + (Point - 1));

    if (FAST_NOT_OK)
      {
	  backwchars(&Number_One);
	  del();
	  return(1);
      }

    backwchars(&Number_One);
    putc('\b',stdout);
    Screen_Col--;
    fast_del();
    tt_delete_char();
    return(0);
}

int backward_delete_char_untabify()
{
    unsigned char *p;
    int n;

    p = CLine->data + (Point - 1);

    /* note that short circuit is assumed to avoid seg fault */
    if (!Point || bobp() || (*p != '\t') || !User_Vars.tab) return backward_delete_char_cmd();

    n = calculate_column() - 1;
    backwchars(&Number_One);
    del();
    n = n - calculate_column();
    ins_char_n_times(' ', n);

    return(1);
}

int previous_line_cmd()
{
   int ret, gc;

   if (CLine == CBuf->beg)
     {
          msg_error(Top_Of_Buffer_Error);
          return(0);
      }

    gc = calculate_column();
    if (!Cursor_Motion) Goal_Column = gc;
    else if (Goal_Column < gc) Goal_Column = gc;

    Cursor_Motion = 1;
    if (!Window->trashed && (CLine != Screen[Window->top - 1].line))
      {
	 ret = 0;
      }
    else ret = 1;
    CLine = CLine->prev;
    point_column(Goal_Column);
    return(ret);
}

int next_line_cmd()
{
   int ret, gc;

   if (CLine == CBuf->end)
     {
	msg_error(End_Of_Buffer_Error);
	return(0);
     }

   gc = calculate_column();
   if (!Cursor_Motion) Goal_Column = gc;
   else if (Goal_Column < gc) Goal_Column = gc;

   Cursor_Motion = 1;
   if (!Window->trashed && (CLine != Screen[Window->top + Window->rows - 2].line))
     {
	ret = 0;
     }
   else ret = 1;
   CLine = CLine->next;
   point_column(Goal_Column);
   return(ret);
}

int next_char_cmd()
{
   Cursor_Motion = 1;
   if (!forwchars(&Number_One))
     {
	msg_error(End_Of_Buffer_Error);
	return(0);
     }
   Goal_Column = calculate_column();
   return(!Point);  /* Point = 0 ==> moved a line */
}

extern Buffer *Paste_Buffer;

void insert_line(Line *line)
{
    line->prev = CLine->prev;
    line->next = CLine;
    if (CLine->prev != NULL)
      {
          CLine->prev->next = line;
      }
    else
      CBuf->beg = line;

    CLine->prev = line;
    if ((line->data)[line->len - 1] != '\n')
      {
          CLine = line;
          Point = line->len;
          ins('\n');
          CLine = line->next;
      }
   mark_buffer_modified(&Number_One);

   Point = 0;
}

int double_line()
{
   int pnt = Point;

   insert_line(dup_line(CLine));
   Point = pnt;
   return(1);
}

int kill_line()
{
    int bol;
    if (eobp())
      {
          msg_error(End_Of_Buffer_Error);
          return(0);
      }

   bol = bolp();

   push_mark(); push_mark();
   if (eolp()) forwchars(&Number_One);
   else
     {
	eol();
	if (bol) forwchars(&Number_One);
     }

   if ((Last_Key_Function == (VOID *) kill_line) && (Paste_Buffer != NULL))
     {
	copy_region_to_buffer(Paste_Buffer);
     }
   else copy_to_pastebuffer();

   delete_region();
   return(1);
}

/* get indent value of current line, n is the column */
unsigned char *get_current_indent(int *n)
{
   unsigned char *p, *pmax;
   int tab;

   tab = User_Vars.tab;
   p = CLine->data;
   pmax = CLine->data + CLine->len;
   *n = 0;
   while((p < pmax) && ((*p == ' ') || (tab && (*p == '\t'))))
     {
	if (*p == '\t')
	  *n = tab * (*n / tab + 1);
	else (*n)++;

	p++;
     }
   return(p);
}

unsigned char *get_indent(int *pt)
{
    int n;
    unsigned char *p;

    p = get_current_indent(&n);

    if (CBuf->modes & C_MODE)
      {
	 push_spot();
	 eol();
	 if (find_matching(')', 50, 0) >= 0)   /* goback 50 lines */
	   {
	      n = calculate_column() - User_Vars.c_indent;
	      p = CLine->data + Point;
	      pop_spot();
	   }
	 else
	   {
	      pop_spot();
	      push_spot();
	      eol();
	      if (find_matching('}', -1, 0) >= 0)
		{
		   p = get_current_indent(&n);
		}
	      pop_spot();
	   }
      }
    if (n < 0) n = 0;
    *pt = n;
    return(p);
}

int trim_whitespace()
{
    unsigned char *p;
    int n, save_point, len;

    len = CLine->len;
    if (len == 0) return(0);

    save_point = Point;
    if (Point == len) Point--;

    /* Note that we save the point before --ing it.  This way the comparison
       made below will effectively restore it if pointing at non whitespace */

    p = CLine->data + Point;
    if (!bolp() && (*p == '\n'))
      {
          Point--;
          p--;
      }

    while ((Point > 0) && ((*p == '\t') || (*p == ' ')))
      {
          Point--;
          p--;
      }

    if (save_point != Point) if (!eolp() && ((*p != ' ') && (*p != '\t')))
      {
          Point++;
          p++;
      }

    n = 0;
    /* this needs to be inlined */
    if (!eobp()) while ((*p == ' ') || (*p == '\t'))
      {
          if (eolp()) break;
          del();
          n++;
      }
    return(1);
}

/* indent line to column n */
void indent_to(int n)
{
    int m;

    get_current_indent(&m);

    if (n != m)
      {
	 Point = 0;
	 trim_whitespace();
	 if (n >= 0) insert_whitespace(&n);
      }
}

int indent_line()
{
    char ch, ch1;
    int n, n1;

    if (CLine == CBuf->beg) return(0);

    push_spot();
    CLine = CLine->prev;
    ch = (char) *get_indent(&n);
    CLine = CLine->next;

    if (CBuf->modes & C_MODE)
      {
	 ch1 = (char) *get_current_indent(&n1);

	 if (n || (ch == '{'))
	   {
	      if (ch1 == '{') n += User_Vars.c_brace;
	      if (ch1 != '}') n += User_Vars.c_indent;
	   }

	 /* else if (ch == '}') n -= User_Vars.c_brace; */

	 /* if (n && (ch1 == '{')) n += User_Vars.c_brace;
	 else if (ch1 == '}') n -= User_Vars.c_indent;*/

	 if (ch1 == '#') n = 0;
      }

    indent_to(n);

    pop_spot();

    /* This moves the cursor to the first non whitspace char if we are
       before it.  Otherwise leave it alone */

    n1 = calculate_column();
    get_current_indent(&n);
    if (n1 < n) point_column(n + 1);
    return(1);
}

int transpose_lines()
{
   if (CLine == CBuf->beg) return(0);
   swap_lines();
   mark_buffer_modified(&Number_One);
   nextline(&Number_Two);
   return(1);
}

int newline_and_indent()
{
    newline();
    indent_line();
    return(1);
}

int ins_char_cmd()
{
    char ch;

    ch = (char) Last_Command_Char;

    if (ch == '\n')
      {
          newline();
          return(1);
      }
    if ((ch == ' ') && (CBuf->modes & WRAP_MODE)
	&& (calculate_column() > User_Vars.wrap_column)) /* Window->width */
      {
          ins(ch);
          wrap_line(0);			/* do not format--- just wrap */
          return(1);
      }

    if (FAST_NOT_OK)
      {
	 ins(ch);
	 if (((ch == '}') || (ch == ')') || (ch == ']'))
	     && !input_pending(2,0)) blink_match(ch);
	 return(1);
      }

    if (!eolp())
      {
	 begin_insert();

	 putc(ch, stdout);
	 end_insert();
      }
   else putc(ch, stdout);
   fast_ins(ch);
   Screen_Col++;
   if ((ch == ')') || (ch == '}') || (ch == ']')) blink_match(ch);
   return(0);
}

int brace_bra_cmd()
{
   int n;
    if ((!(CBuf->modes & C_MODE)) || (!eolp())
         || (Point && (*(CLine->data + (Point - 1)) == '\'')))
      {
         return ins_char_cmd();
      }

    if (*get_current_indent(&n) > ' ') newline();
    indent_line();
    ins('{');
    indent_line();
    newline_and_indent();
    return(1);
}

int brace_ket_cmd()
{
   int n;
   int pnt = Point;
   unsigned char *prev;

   if ((CBuf->modes & C_MODE) == 0) return ins_char_cmd();
   
   prev = CLine->data + (Point - 1);
   skip_whitespace();
   if (!eolp() || (pnt && ((*prev == '"') || (*prev == '\''))))
     {
	Point = pnt;
	return ins_char_cmd();
     }

   if ((char) *get_current_indent(&n) > ' ') newline();
   ins('}');
   indent_line();
   Window->trashed = 1;
   update((Line *) NULL);
   blink_match('}');
   newline_and_indent();
   return(1);
}

int eol_cmd()
{
    if (!eolp())
      eol();
    trim_whitespace();
    return(1);
}

int c_comment()
{
   int t = 3;
   eol();
   ins_chars((unsigned char *) "   /*  */",9);
   backwchars(&t);
   return(1);
}

int sys_spawn_cmd()
{
   lang_run_hooks("suspend_hook", NULL);
   if (Lang_Error) return(0);
   reset_tty();
   reset_display();
   fflush(stdout);
   sys_suspend();
   init_tty();
   init_display();
   cls();
   lang_run_hooks("resume_hook", NULL);
   check_buffers();
   return(1);
}

int quoted_insert()
{
   if (*Error_Buffer || KeyBoard_Quit) return(0);
   if (('\n' == (Last_Command_Char = jed_getkey())) && (IS_MINIBUFFER))
     {
	msg_error("Not allowed!");
	return(1);
     }

   KeyBoard_Quit = 0;
   return(ins_char_cmd());
}

int quit_jed(void)
{
   reset_display();
   reset_tty();
   exit(0);
   return(1);
}

/* I should try it like emacs--- if prefix argument, then save all without user
   intervention */
int save_some_buffers(void)
{
   Buffer *b, *tmp;
   int ans = 0;
   char msg[255];
   int err;

   b = CBuf;
   do
     {
	if ((b->flags & AUTO_SAVE_BUFFER) && (b->flags & BUFFER_TRASHED)
	    && (*b->file))
	  {
	     while(1)
	       {
		  sprintf(msg,"Buffer %s not saved. Save it? (y/n)", b->name);
		  message(msg);
		  flush_message();
		  ans = getkey();
		  if ((ans == 'y') || (ans == 'Y'))
		    {
		       tmp = CBuf;
		       CBuf = b;
		       widen();
		       err = write_file_with_backup(b->dir, b->file);
		       CBuf = tmp;
  		       if (err < 0) return(-1);
		       b->flags &= ~BUFFER_TRASHED;
		       b->flags |= AUTO_SAVE_BUFFER;
		       b->hits = 0;
		       break;
		    }
		  else if ((ans == 'n') || (ans == 'N'))
		    {
		       /* disabling this */
		       /* auto_save_buffer(b); */
		       break;
		    }
		  else if (ans == 7)
		    {
		       KeyBoard_Quit = 1;

		       /* warning--- bug here if user edits file at
		       startup and forgets to save it then aborts. */
		       return(-1);
		    }
		  else beep();
	       }
	  }
	b = b->next;
     }
   while (b != CBuf);
   message(" ");
   return(1);
}

int exit_jed()
{
   lang_run_hooks("exit_hook", NULL);
   if (Lang_Error) return(0);

   if (save_some_buffers() > 0) quit_jed();
   return(1);
}

int pageup_cmd()
{
    int t,b, upd;
    Line *top;

   b = Window->rows - 2;
   if (b < 0) return(0);
   b = (!b ? 1 : b);
   t = Window->top - 1;
   if (!Window->trashed)
     {
	if (CLine->prev == NULL)
	  {
	     if (!Point) msg_error(Top_Of_Buffer_Error);
	     Point = 0;
	     return(1);
	  }
	CLine = Screen[t].line;
	prevline(&b);
	top = CLine;
	CLine = Screen[t].line;
	Window->trashed = 1;
	update(top);
	upd = 0;
     }
   else if (0 == prevline(&b))
     {
	if (!Point) msg_error(Top_Of_Buffer_Error);
	Point = 0;
	return(1);
     }
   else upd = 1;
   
   if (Point >= CLine->len) Point = CLine->len - 1;
   if (Point < 0) Point = 0;
   return(upd);
}

int pagedown_cmd()
{
   int b, upd;
   Line *top;
   
   if (Window->rows < 2) return(0);
   b = Window->rows + Window->top - 2;
   if (eobp())
     {
	msg_error(End_Of_Buffer_Error);
	return(0);
     }
   
   if (!Window->trashed && (Screen[b].line == NULL))
     {
	eob();
	return(1);
     }
   
   upd = 1;
   if (!Window->trashed)
     {
	top = CLine = Screen[b - 1].line;
	Window->trashed = 1;
	update(top);
	upd = 0;
     }
   else
     {
	if (b > 2) b = Window->rows - 2; else b = 1;
	nextline(&b);
     }
   
   if (Point >= CLine->len) Point = CLine->len - 1;
   if (Point < 0) Point = 0;
   return(upd);
}

int scroll_right()
{
   if (Window->column == 1) return(0);

   if ((Window->column = Window->column - Window->width / 2) < 1)
     {
	Window->column = 1;
     }

   touch_window();
   return(1);
}

int scroll_left()
{
    Window->column = Window->column + Window->width / 2;
    touch_window();
    return(1);
}

/* goto to column c, returns actual column */
int goto_column1(int c)
{
   int c1;
   if (c <= 0) return(0);  /* not valid */
   eol();
   c1 = calculate_column();
   if (c1 > c)
     {
	point_column(c);
	c1 = calculate_column();
     }
   return(c1);
}

/* move cursor to column c adding spaces if necessary */
void goto_column(int *c)
{
   int c1;
   if (*c <= 0) return;
   c1 = *c - goto_column1(*c);
   insert_whitespace(&c1);
}

/* does not leave current line */
unsigned int skip_whitespace()
{
   unsigned char *p, *pmax;

   if (CLine->len == 0) return('\n');

   p = CLine->data + Point;
   eol();
   pmax = CLine->data + Point;
   while(p < pmax)
     {
	if ((*p != ' ') && (*p != '\t')) break;
	p++;
     }
   Point = (int) (p - CLine->data);
   return(*p);
}

#define upcase(ch)\
((!Case_Sensitive && ((ch) <= 'z') && ((ch) >= 'a')) ? (ch) & 0xDF : (ch))


int STRNCMPI(char *a, char *b, int n)
{
   while (n && *a && *b && (upcase(*a) == upcase(*b))) a++, b++, n--;
   return(!n || ((*a == 0) && (*b == 0)));
}

/* returns < 0 if what is smaller than current thing, 0 if matched, pos otherwise */
int looking_at(char *what)
{
   int n = strlen(what);
   int n1 = CLine->len - Point;
   unsigned char *p = CLine->data + Point;

   if (n > n1) return(0);
   return(STRNCMPI(what, (char *) p, n));
}

static void make_lut(int *lut, char *range, int reverse)
{
   int i, r1, r2;
   
   for (i = 0; i < 128; i++) lut[i] = reverse;
   
   while (*range)
     {
	r1 = upcase(*range);
	if (*(range + 1) == '-')
	  {
	     range += 2;
	     r2 = upcase(*range);
	  }
	else r2 = r1;
	
	for (i = r1; i <= r2; i++) lut[i] = !lut[i];
	if (*range) range++;
     }
}

   

int skip_chars1(char *range, int reverse)
{
   int lut[128];
   unsigned char *p, *pmax, ch;
   int count, pnt;
   
   if (eolp()) return(0);

   make_lut(lut, range, reverse);
   pnt = Point;
   p = CLine->data + Point;
   pmax = CLine->data + CLine->len;
   count = 0;

   while (p < pmax)
     {
	ch = upcase(*p);
	if (0 == lut[ch]) break;
	count++;
	p++;
     }
   Point = Point + count;
   return(Point - pnt);
}

int bskip_chars1(char *range, int reverse)
{
   int lut[128];
   unsigned char *p, ch;
   int count;
   
   if (!Point) return(0);

   make_lut(lut, range, reverse);
   p = CLine->data + (Point - 1);
   count = 0;

   while (p >= CLine->data)
     {
	ch = upcase(*p);
	if (0 == lut[ch]) break;
	count++;
	p--;
     }
   Point = Point - count;
   return(count);
}

int skip_chars (char *range)
{
   return(skip_chars1(range, 0));
}
int bskip_chars (char *range)
{
   return(bskip_chars1(range, 0));
}

#if 0
/* skip past chars in range s:  ie, s = "az" means skip those in that range */
/* if a space occurs, skip whitespace */
int skip_chars(char *s)
{
   unsigned char *p, *pmax, min, max, ch;
   int count, pnt;

   if (eolp()) return(0);

   pnt = Point;
   if (*s == ' ')
     {
	skip_whitespace();
	s++;
     }
   if (! *s) return(Point - pnt);

   ch = *s++; min = upcase(ch);
   ch = *s++; max = upcase(ch);
   if (max < min)
     {
	s--;
	max = min;
     }

   p = CLine->data + Point;
   pmax = CLine->data + CLine->len;
   count = 0;

   while (p < pmax)
     {
	ch = upcase(*p);
	if ((ch < min) || (ch > max)) break;
	count++;
	p++;
     }
   Point = Point + count;
   if (*s == ' ') skip_whitespace();
   return(Point - pnt);
}
#endif
